/**
 * @file      nios32_isa.cpp
 * @author    Renato Herebia
 *
 *            The ArchC Team
 *            http://www.archc.org/
 *
 *            Universidade Católica Dom Bosco - UCDB 
 *
 * @version   1.0
 * @date      Mon, 19 Jun 2006 15:33:28 -0300
 * 
 * @brief     The ArchC Altera/NIOS cycle-accurate model.
 * 
 * @attention Copyright (C) 2002-2006 --- The ArchC Team
 * 
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 * 
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

//#include "archc.H"
//#include "ac_types.H"
//#include "ac_parms.H"
//#include "nios32_isa.H"
//#include "ac_isa_init.cpp"
#include <math.h>

//moving from archc 1.6 to archc 2.0
#include "nios32_isa.H"
#include "nios32_isa_init.cpp"
#include "nios32_bhv_macros.H"

using namespace nios32_parms;

//! Behavior executed before simulation begins.
void ac_behavior(begin) {
	return;
}

//! Behavior executed after simulation ends.
void ac_behavior(end) {
	return;
}

/** Comportamento de uma instrução no pipeline.
 *
 */
void ac_behavior( instruction ){

	switch( stage ) {
		case id_pipe_IF:
			cout << "\n------------IF------------\n";
			ac_pc += 4;
			IF_ID.npc = ac_pc;
			break;
	
		case id_pipe_ID:
			cout << "\n------------ID------------\n";
			break;
	
		case id_pipe_EX:
			cout << "\n------------EX------------\n";
			break;

		case id_pipe_MEM:
			cout << "\n------------MEM------------\n";
			MEM_WB.rdest = EX_MEM.rdest;
			MEM_WB.wdata = EX_MEM.wdata;
			MEM_WB.regwrite = EX_MEM.regwrite;
			break;
	
		case id_pipe_WB:
			cout << "\n------------WB------------\n";

			/* Execute write back when allowed */
			if (MEM_WB.regwrite == 1) {
				// Register 0 is never written
				if (MEM_WB.rdest != 0)
					cout << "\n\t%r" << MEM_WB.rdest << " = " << MEM_WB.wdata << "\n";
					RB.write(MEM_WB.rdest, MEM_WB.wdata);
			}
			break;
	}
};

/** Implements the behavior of decode stage of format i4w
 *
 */
void ac_behavior ( Format_i4w ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			//ID_EX.data1 = ;
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm4w;
			//ID_EX.ra = ;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format i6v
 *
 */
void ac_behavior ( Format_i6v ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			//ID_EX.data1 = ;
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm6v;
			//ID_EX.ra = ;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format i8v
 *
 */
void ac_behavior ( Format_i8v ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			//ID_EX.data1 = ;
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm8v;
			//ID_EX.ra = ;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format i9
 *
 */
void ac_behavior ( Format_i9 ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			//ID_EX.data1 = ;
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm9;
			//ID_EX.ra = ;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format i10
 *
 */
void ac_behavior ( Format_i10 ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			//ID_EX.data1 = ;
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm10;
			//ID_EX.ra = ;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format i11.
 *
 */
void ac_behavior ( Format_i11 ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			IF_ID.opcode = op5; //is it right? *****
			//ID_EX.data1 = RB[ra];
			//ID_EX.data2 = RB[rb];
			ID_EX.imm = imm11;
			//ID_EX.ra = ra;
			//ID_EX.rb = rb;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format Ri1u
 *
 */
void ac_behavior ( Format_Ri1u ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			ID_EX.data1 = RB[ra];
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm1u;
			ID_EX.ra = ra;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format Ri2u
 *
 */
void ac_behavior ( Format_Ri2u ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			ID_EX.data1 = RB[ra];
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm2u;
			ID_EX.ra = ra;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format Ri4
 *
 */
void ac_behavior ( Format_Ri4 ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			ID_EX.data1 = RB[ra];
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm4;
			ID_EX.ra = ra;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format Ri5
 *
 */
void ac_behavior ( Format_Ri5 ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			ID_EX.data1 = RB[ra];
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm5;
			ID_EX.ra = ra;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format Ri6
 *
 */
void ac_behavior ( Format_Ri6 ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			ID_EX.data1 = RB[ra];
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm6;
			ID_EX.ra = ra;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format Ri8
 *
 */
void ac_behavior ( Format_Ri8 ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			ID_EX.data1 = RB[ra];
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm8;
			ID_EX.ra = ra;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format RPi5
 *
 */
void ac_behavior ( Format_RPi5 ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			ID_EX.data1 = RB[ra];
			//ID_EX.data2 = ;
			ID_EX.imm = 0x00000000 | imm5;
			ID_EX.ra = ra;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format RR.
 *
 */
void ac_behavior ( Format_RR ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}
};

/** Implements the behavior of decode stage of format Rw
 *
 */
void ac_behavior ( Format_Rw ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			ID_EX.data1 = RB[ra];
			//ID_EX.data2 = ;
			//ID_EX.imm = ;
			ID_EX.ra = ra;
			//ID_EX.rb = ;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Implements the behavior of decode stage of format w
 *
 */
void ac_behavior ( Format_w ) {

	switch ( stage ) {
		case id_pipe_IF:
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;
			//ID_EX.data1 = ;
			//ID_EX.data2 = ;
			//ID_EX.imm = ;
			//ID_EX.ra = ;
			//ID_EX.rb = ;
			ID_EX.regwrite = 0;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ABS (Absolute Value) from format Rw
 * status: finished, tested [Amended]
 */
void ac_behavior ( abs ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tabs\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = abs(ID_EX.ra);
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}
};

/** Instruction ADD from format RR.
 * not finished, not tested [Amended]
 */
void ac_behavior ( add ) {

	int temp;
	int carry;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tadd\n");
			break;
		case id_pipe_ID:
			//ID_EX.npc = IF_ID.npc;

			//fowarding for register 1
			if ( MEM_WB.rdest == ra ) {
				cout << "\n\tfowarding ra from WB\n";
				ID_EX.data1 = MEM_WB.wdata;
			}
			else if ( EX_MEM.rdest == ra ) {
				cout << "\n\tfowarding ra from MEM\n";
				ID_EX.data1 = EX_MEM.wdata;
			}
			else {
				cout << "\n\tra from RB\n";
				ID_EX.data1 = RB[ra];
			}

			//fowarding for register 2
			if ( MEM_WB.rdest == rb ) {
				cout << "\n\tfowarding rb from WB\n";
				ID_EX.data2 = MEM_WB.wdata;
			}
			else if ( EX_MEM.rdest == rb ) {
				cout << "\n\tfowarding rb from MEM\n";
				ID_EX.data2 = EX_MEM.wdata;
			}
			else {
				cout << "\n\trb from RB\n";
				ID_EX.data2 = RB[rb];
			}

			//ID_EX.imm = 0;
			ID_EX.ra = ra;
			ID_EX.rb = rb;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = ID_EX.ra + ID_EX.rb;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			
			//control registers
			temp = EX_MEM.wdata & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;

			if ( (ID_EX.ra & 0x80000000) == 1 && (ID_EX.rb & 0x80000000) == 1 )
				ctl0.v = 1;
			else 
				ctl0.v = 0;

			if ( EX_MEM.wdata == 0 ) 
				ctl0.z = 1;
			else 
				ctl0.z = 0;

			carry = EX_MEM.wdata >> 31;
						
			if( carry == 1) 
				ctl0.c = 1;
			else 
				ctl0.c = 0;

			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}
};

/** Instruction ADDI (Add Imediate) from format Ri5
 * status: finished, not tested [Amended]
 */
void ac_behavior ( addi ) {

	int temp;
	int carry;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\taddi\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			if ( k.k == 0 ) {//not preceded by PFX or PFXIO
				EX_MEM.imm = ID_EX.imm;
				EX_MEM.wdata = ID_EX.ra + ID_EX.imm;
				EX_MEM.rdest = ID_EX.ra;
				EX_MEM.regwrite = ID_EX.regwrite;
				EX_MEM.memread = ID_EX.memread;
				EX_MEM.memwrite = ID_EX.memwrite;
			}
			else {//preceded by PFX or PFXIO
				EX_MEM.imm = ID_EX.imm;
				EX_MEM.wdata = ID_EX.ra + (ID_EX.imm | (k.k<<5));
				EX_MEM.rdest = ID_EX.ra;
				EX_MEM.regwrite = ID_EX.regwrite;
				EX_MEM.memread = ID_EX.memread;
				EX_MEM.memwrite = ID_EX.memwrite;
			}

			//control registers
			temp = EX_MEM.wdata & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;

			if ( (ID_EX.ra & 0x80000000) == 1)
				ctl0.v = 1;
			else 
				ctl0.v = 0;

			if ( EX_MEM.wdata == 0 )
				ctl0.z = 1;
			else
				ctl0.z = 0;

			carry = EX_MEM.wdata >> 31;
						
			if( carry == 1) ctl0.c = 1;
			else ctl0.c = 0;

			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction AND (Bitwise Logical AND) not preceded by PFX from format RR
 * status: finished, not tested [Amended]
 */
void ac_behavior ( and1 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tand not preceded by PFX\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = ID_EX.ra & ID_EX.rb;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;

			//control registers
			temp = EX_MEM.wdata & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;

			if ( EX_MEM.wdata == 0 )
				ctl0.z = 1;
			else
				ctl0.z = 0;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction AND (Bitwise Logical AND) preceded by PFX from format Ri5
 * status: finished, not tested [Amended]
 * problem with identification! ??
 */
void ac_behavior ( and2 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tand preceded by PFX");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			EX_MEM.imm = ID_EX.imm;
			EX_MEM.wdata = ID_EX.ra & ( EX_MEM.imm | (k.k << 5) );
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;

			//control registers
			temp = EX_MEM.wdata & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;

			if ( EX_MEM.wdata == 0 ) 
				ctl0.z = 1;
			else
				ctl0.z = 0;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ANDN (Bitwise Logical AND NOT) not preceded by PFX from format RR
 * status: finished, not tested
 */
void ac_behavior ( andn1 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tandn not preceded by pfx\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = ID_EX.ra & ~ID_EX.data2;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;

			//control registers
			temp = EX_MEM.wdata & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;

			if ( EX_MEM.wdata == 0 )
				ctl0.z = 1;
			else
				ctl0.z = 0;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ANDN (Bitwise Logical AND NOT) preceded by PFX from format Ri5
 * status: finished, not tested
 */
void ac_behavior ( andn2 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tandn preceded by pfx\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			EX_MEM.imm = k.k<<5 | ID_EX.imm;
			EX_MEM.wdata = ID_EX.ra & ~EX_MEM.imm;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;

			//control registers
			temp = EX_MEM.wdata & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;
			
			if ( EX_MEM.wdata == 0 )
				ctl0.z = 1;
			else
				ctl0.z = 0;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ASR (Arithmetic Shift Right) from format RR
 * status: finished, not tested
 */
void ac_behavior ( asr ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tasr\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = ID_EX.ra>>(ID_EX.rb & 0x0000001F);
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ASRI (Arithmetic Shift Right Immediate) from format Ri5
 * status: finished, tested [Amended]
 */
void ac_behavior ( asri ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tasri\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			EX_MEM.imm = ID_EX.imm;
			EX_MEM.wdata = ID_EX.ra>>EX_MEM.imm;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction BGEN (Bit Generate) from format Ri5
 * status: finished, not tested
 */
void ac_behavior ( bgen ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tbgen\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = pow(2,ID_EX.imm);
			EX_MEM.rdest = ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction BR (Branch) from format i11
 * status: finished, not tested
 */
void ac_behavior ( br ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tbr\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ID_EX.regwrite;
			//EX_MEM.memread = ID_EX.memread;
			//EX_MEM.memwrite = ID_EX.memwrite;

			//increments pc
			ac_pc = ac_pc + (ID_EX.imm+1)<<1;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction BSR (Branch To Subroutine) from format i11
 * status: finished, not tested
 */
void ac_behavior ( bsr ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tbsr\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			RB[15] = (ac_pc + 4)>>1; // return address
			ac_pc = ac_pc + (ID_EX.imm + 1)<<1;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction CALL (Call Subroutine) from format Rw
 * status: finished, not tested
 */
void ac_behavior ( call ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tcall\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			RB[15] = (ac_pc+4)>>1;
			ac_pc = ID_EX.ra<<1;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction CMP (Compare) from format RR
 * status: finished, not tested [Amended]
 */
void ac_behavior ( cmp ) {

	int result, temp, carry;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tcmp\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			result = ID_EX.ra - ID_EX.rb;

			temp = result & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;

			if ( (ID_EX.ra & 0x80000000) == 0 && (ID_EX.rb & 0x80000000) == 1 )
				ctl0.v = 1;
			else 
				ctl0.v = 0;

			if ( result == 0 )
				ctl0.z = 1;
			else 
				ctl0.z = 0;

			carry = result >> 31;
						
			if( carry == 1) 
				ctl0.c = 1;
			else 
				ctl0.c = 0;

			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction CMPI (Compare Immediate) from format Ri5
 * status:  finished, not tested [Amended]
 */
void ac_behavior ( cmpi ) {

	int extended, result, temp, carry;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tcmpi\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			if ( k.k == 0 ) //not preceded by PFX
				result = ID_EX.ra - ID_EX.imm;
			else {//preceded by PFX
				extended = k.k<<5 + ID_EX.imm;

				result = ID_EX.ra - extended;
			}

			temp = result & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;

			if ( (ID_EX.ra & 0x80000000) == 0 ) //&& (ID_EX.data2 & 0x80000000) == 1
				ctl0.v = 1;
			else 
				ctl0.v = 0;

			if ( result == 0 )
				ctl0.z = 1;
			else 
				ctl0.z = 0;

			carry = result >> 31;
						
			if( carry == 1) 
				ctl0.c = 1;
			else 
				ctl0.c = 0;

			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction EXT16D (Half-Word Extract (Dynamic)) from format RR
 * status: finished, not tested
 */
void ac_behavior ( ext16d ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\text16d\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;

			temp = ID_EX.rb & 0x00000002;

			temp = temp >> 1;

			if ( temp == 1 )
				EX_MEM.wdata = ID_EX.ra & 0x0000FFFF;
			else 
				EX_MEM.wdata = (ID_EX.ra & 0xFFFF0000) >> 16;
			
			EX_MEM.rdest = ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction EXT16S (Half-Word Extract (Static)) from format Ri1u
 * status: finished, not tested
 */
void ac_behavior ( ext16s ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\text16s\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ID_EX.imm;

			if ( ID_EX.imm == 1 )
				EX_MEM.wdata = ID_EX.ra & 0x0000FFFF;
			else 
				EX_MEM.wdata = (ID_EX.ra & 0xFFFF0000) >> 16;

			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction EXT8D (Byte-Extract (Dynamic)) from format RR
 * status: finished, not tested
 */
void ac_behavior ( ext8d ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\text8d\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
		
			temp = ID_EX.rb & 0x00000003;

			switch ( temp ) {
				case 0:
					EX_MEM.wdata = ID_EX.ra & 0x000000FF;
					break;
				case 1:
					EX_MEM.wdata = (ID_EX.ra & 0x0000FF00)>>8;
					break;
				case 2:
					EX_MEM.wdata = (ID_EX.ra & 0x00FF0000)>>16;
					break;
				case 3:
					EX_MEM.wdata = (ID_EX.ra & 0xFF000000)>>24;
					break;
			}

			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction EXT8S (Byte-Extract (Static)) from format Ri2u
 * status: finished, not tested
 */
void ac_behavior ( ext8s ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\text8s\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;

			switch ( ID_EX.imm ) {
				case 0:
					EX_MEM.wdata = ID_EX.ra & 0x000000FF;
					break;
				case 1:
					EX_MEM.wdata = (ID_EX.ra & 0x0000FF00)>>8;
					break;
				case 2:
					EX_MEM.wdata = (ID_EX.ra & 0x00FF0000)>>16;
					break;
				case 3:
					EX_MEM.wdata = (ID_EX.ra & 0xFF000000)>>24;
					break;
			}

			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction FILL16 (Half-Word Fill) from format Rw
 * status: finished, not tested
 */
void ac_behavior ( fill16 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tfill16\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;

			temp = ID_EX.data2 & 0x0000FFFF;

			EX_MEM.wdata = (temp<<16)+temp;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction FILL8 (Byte-Fill) from format Rw
 * status: finished, not tested
 */
void ac_behavior ( fill8 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tfill8\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;

			temp = ID_EX.data2 & 0x000000FF;

			EX_MEM.wdata = (temp<<24) + (temp<<16) + (temp<<8) + temp;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction IF0 (Execute if Register Bit is 0) from format Ri5
 * status: finished, not tested
 */
void ac_behavior ( if0 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tif0\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			temp = ra & (int)pow(2,ID_EX.imm);

			temp = temp >> (ID_EX.imm);

			if ( temp == 1 ) 
				if ( IF_ID.opcode == 0x13 || IF_ID.opcode == 0x12 ) //verify the next instruction
					ac_pc = ac_pc + 8;
				else 
					ac_pc = ac_pc + 4;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction IF1 (Execute if Register Bit is 1) from format Ri5
 * status: finished, not tested
 */
void ac_behavior ( if1 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tif1\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			temp = ra & (int)pow(2,ID_EX.imm);

			temp = temp >> (ID_EX.imm);

			if ( temp == 0 ) 
				if ( IF_ID.opcode == 0x13 || IF_ID.opcode == 0x12 ) //verify the next instruction
					ac_pc = ac_pc + 8;
				else 
					ac_pc = ac_pc + 4;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction IFRNZ (Execute if Register is not Zero) from format Rw
 * status: finished, not tested
 */
void ac_behavior ( ifrnz ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tifrnz\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			if ( ID_EX.ra == 0 ) 
				if ( IF_ID.opcode == 0x13 || IF_ID.opcode == 0x12 ) //verify the next instruction
					ac_pc = ac_pc + 8;
				else 
					ac_pc = ac_pc + 4;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction IFRZ (Execute if Register is Zero) from format Rw
 * status: finished, not tested
 */
void ac_behavior ( ifrz ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tifrz\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			if ( ID_EX.ra != 0 ) 
				if ( IF_ID.opcode == 0x13 || IF_ID.opcode == 0x12 ) //verify the next instruction
					ac_pc = ac_pc + 8;
				else 
					ac_pc = ac_pc + 4;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction IFS (Conditionally Execute Next Instruction) from format i4w
 * status: finished, not tested
 */
void ac_behavior ( ifs ) {

	bool flag;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tifs\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			flag = false;

			switch ( ID_EX.imm ) {
				case 0x0:
					if ( ctl0.c == 0 ) flag = true;
					break;
				case 0x1:
					if ( ctl0.c == 1 ) flag = true;
					break;
				case 0x2:
					if ( ctl0.z == 0 ) flag = true;
					break;
				case 0x3:
					if ( ctl0.z == 1 ) flag = true;
					break;
				case 0x4:
					if ( ctl0.n == 0 ) flag = true;
					break;
				case 0x5:
					if ( ctl0.n == 1 ) flag = true;
					break;
				case 0x6:
					if ( ctl0.n xor ctl0.v ) flag = true;
					break;
				case 0x7:
					if ( (ctl0.n xor ctl0.v) == 0 ) flag = true;
					break;
				case 0x8:
					if ( (ctl0.z or (ctl0.n xor ctl0.v)) == 0 ) flag = true;
					break;
				case 0x9:
					if ( (ctl0.z or (ctl0.n xor ctl0.v)) == 1 ) flag = true;
					break;
				case 0xa:
					if ( ctl0.v == 0 ) flag = true;
					break;
				case 0xb:
					if ( ctl0.v == 1 ) flag = true;
					break;
				case 0xc:
					if ( (ctl0.c or ctl0.z) == 0 ) flag = true;
					break;
				case 0xd:
					if ( (ctl0.c or ctl0.z) == 1 ) flag = true;
					break;
				
			}

			if ( flag )
				if ( IF_ID.opcode == 0x13 || IF_ID.opcode == 0x12 ) //verify the next instruction
					ac_pc = ac_pc + 8;
				else 
					ac_pc = ac_pc + 4;

			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction JMP (Computed Jump) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( jmp ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tjmp\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite = ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			ac_pc = ID_EX.ra << 0x00000001;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction LD (Load 32-Bit Data From Memory) from format RR
 * status: finished, not tested [Amended]
 */
void ac_behavior ( ld ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tld\n");
			break;
		case id_pipe_ID:

			//fowarding for register 2
			if ( MEM_WB.rdest == rb ) {
				cout << "\n\tfowarding rb from WB\n";
				ID_EX.data2 = MEM_WB.wdata;
			}
			else if ( EX_MEM.rdest == rb ) {
				cout << "\n\tfowarding rb from MEM\n";
				ID_EX.data2 = EX_MEM.wdata;
			}
			else {
				cout << "\n\trb from RB\n";
				ID_EX.data2 = RB[rb];
			}

			//ID_EX.imm = 0;
			//ID_EX.ra = ra;
			ID_EX.rb = rb;
			ID_EX.regwrite = 1;
			ID_EX.memread = 1;
			ID_EX.memwrite = 0;			
			
			break;
		case id_pipe_EX:

			if(k.k == 0){
				//EX_MEM.imm = ;
				EX_MEM.wdata = ID_EX.rb && 0x80000000;
				EX_MEM.rdest = ID_EX.ra;
				EX_MEM.regwrite = ID_EX.regwrite;
				EX_MEM.memread = ID_EX.memread;
				EX_MEM.memwrite = ID_EX.memwrite;
			}else{
				//EX_MEM.imm = ;
				EX_MEM.wdata = (ID_EX.rb + (k.k * 4)) && 0x8000000;
				EX_MEM.rdest = ID_EX.ra;
				EX_MEM.regwrite = ID_EX.regwrite;
				EX_MEM.memread = ID_EX.memread;
				EX_MEM.memwrite = ID_EX.memwrite;
			}

			break;
		case id_pipe_MEM:
			MEM_WB.wdata = EX_MEM.wdata;
			break;
		case id_pipe_WB:
			MEM_WB.wdata = MEM_WB.wdata;
			break;
	}

};

/** Instruction LDP (Load 32-Bit Data From Memory (Pointer Addressing Mode)) from format RPi5
 * status: not finished, not tested
 */
void ac_behavior ( ldp ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tldp\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			//EX_MEM.regwrite ;
			//EX_MEM.memread = ;
			//EX_MEM.memwrite = ;

			break;
		case id_pipe_MEM:
			
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction LDS (Load 32-Bit Data From Memory (Stack Addressing Mode)) from format Ri8
 * status: not finished, not tested
 */
void ac_behavior ( lds ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tlds\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction LRET (Equivalent to JMP %o7) from format Rw
 * status: finished, not tested
 */
void ac_behavior ( lret ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tlret\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
	
			ac_pc = RB[15] << 1;

			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction LSL (Logical Shift Left) from format RR
 * status: not finished, not tested
 */
void ac_behavior ( lsl ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tlsl\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction LSLI (Logical Shift Left Immediate) from format Ri5
 * status: not finished, not tested
 */
void ac_behavior ( lsli ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tlsli\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction LSR (Logical Shift Right) from format RR
 * status: not finished, not tested
 */
void ac_behavior ( lsr ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tlsr\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction LSRI (Logical Shift Right Immediate) from format Ri5
 * status: not finished, not tested
 */
void ac_behavior ( lsri ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tlsri\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction MOV (Register-to-Register Move) from format RR
 * status: finished, not tested [Amended]
 */
void ac_behavior ( mov ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tmov\n");
			break;
		case id_pipe_ID:

			//fowarding for register rb
			if ( MEM_WB.rdest == rb ) {
				cout << "\n\tfowarding rb from WB\n";
				ID_EX.data2 = MEM_WB.wdata;
			}
			else if ( EX_MEM.rdest == rb ) {
				cout << "\n\tfowarding rb from MEM\n";
				ID_EX.data2 = EX_MEM.wdata;
			}
			else {
				cout << "\n\trb from RB\n";
				ID_EX.data2 = RB[rb];
			}

			//ID_EX.imm = 0;
			//ID_EX.ra = ra;
			ID_EX.rb = rb;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;

			break;
		case id_pipe_EX:
			//EX_MEM.imm = 0;
			EX_MEM.wdata = ID_EX.rb;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction MOVHI (Move Immediate Into High Half-Word) from format Ri5
 * status: not finished, not tested
 */
void ac_behavior ( movhi ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tmovhi\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction MOVI (Move Immediate) from format Ri5
 * status: not finished, not tested [Amended]
 */
void ac_behavior ( movi ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tmovi\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			if(k.k == 0){//não precede PFX
				EX_MEM.imm = ( 0x0000000 | (k.k << imm5) );
				EX_MEM.wdata = ID_EX.imm;
				EX_MEM.rdest = ID_EX.ra;
				EX_MEM.regwrite = ID_EX.regwrite;
				EX_MEM.memread = ID_EX.memread;
				EX_MEM.memwrite = ID_EX.memwrite; 
			}
			else{//precede PFX
				EX_MEM.imm = ( 0x0000000 | (k.k << imm5) );
				EX_MEM.wdata = ID_EX.imm;
				EX_MEM.rdest = ID_EX.ra;
				EX_MEM.regwrite = ID_EX.regwrite;
				EX_MEM.memread = ID_EX.memread;
				EX_MEM.memwrite = ID_EX.memwrite;
			}			
			break;
		case id_pipe_MEM:
			MEM_WB.rdest = EX_MEM.rdest;
			MEM_WB.wdata = EX_MEM.wdata;
			MEM_WB.regwrite = EX_MEM.regwrite;
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction MSTEP (Multiply-Step) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( mstep ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tmstep\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction MUL (Multiply) from format Rw
 * status: finished, not tested
 */
void ac_behavior ( mul ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tmul\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = (ctl2.lo_limit & 0x0000.ffff) * (ID_EX.ra & 0x0000.ffff);
			//EX_MEM.rdest = ctl2.lo_limit;
			EX_MEM.regwrite = 1;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction NEG (Arithmetic Negation) from format Rw
 * status: finished, not tested [Amended]
 */
void ac_behavior ( neg ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tneg\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = 0 - ID_EX.ra;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = ID_EX.memread;
			EX_MEM.memwrite = ID_EX.memwrite;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction NOP (Equivalent to MOV %g0, %g0) from format RR
 * status: finished, not tested [Amended]
 */
void ac_behavior ( nop ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tnop\n");
			break;
		case id_pipe_ID:
			ID_EX.imm = 0;
			ID_EX.ra = 0;
			ID_EX.rb = 0;
			ID_EX.data1 = 0;
			ID_EX.data2 = 0;	
			ID_EX.regwrite = 0;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;		
			break;
		case id_pipe_EX:
			EX_MEM.imm = 0;
			EX_MEM.wdata = 0;
			EX_MEM.rdest = 0;
			EX_MEM.regwrite = 0;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			break;
		case id_pipe_MEM:
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			MEM_WB.wdata = 0;
			break;
		case id_pipe_WB:
			EX_MEM.regwrite = 0;
			MEM_WB.wdata = 0;
			break;
	}

};

/** Instruction NOT (Logical Not) from format Rw
 * status: finished, not tested [Amended]
 */
void ac_behavior ( not ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tnot\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = ~ ID_EX.ra;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction OR (Bitwise Logical OR) not preceded by PFX from format RR
 * status: finished, not tested [Amended]
 */
void ac_behavior ( or1 ) {
	
	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tor not preceded by pfx\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			
			//EX_MEM.imm = ;
			EX_MEM.wdata = ID_EX.ra || ID_EX.rb;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			
			temp = EX_MEM.wdata & 0x8000000;
			temp = temp >> 31;

			if(temp == 1)
				ctl0.n = 1;
			else 	ctl0.n = 0;

			if(EX_MEM.wdata == 0)
				ctl0.z = 1;
			else 	ctl0.z = 0;

			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction OR (Bitwise Logical OR) preceded by PFX from format Ri5
 * status: finished, not tested [Amended]
 */
void ac_behavior ( or2 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tor preceded by pfx\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			EX_MEM.imm = ID_EX.imm;
			EX_MEM.wdata = ID_EX.ra || ( ID_EX.imm | (k.k << 5) );
			EX_MEM.rdest = ID_EX.data1;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;

			temp = EX_MEM.wdata & 0x8000000;
			temp = temp >> 31;

			if( temp == 1)
				ctl0.n = 1;
			else 	ctl0.n = 0;

			if(EX_MEM.wdata != 0)
				ctl0.z = 0;
			else	ctl0.z = 1;
			
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction PFX (Prefix) from format i11.
 * status: finished, tested [Amended]
 */
void ac_behavior ( pfx ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\t\tpfx\n");
			break;
		case id_pipe_ID:
			
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;

			//control registers
			k.k = ID_EX.imm;

			break;
		case id_pipe_MEM:
			//MEM_WB.rdest = ;
			//MEM_WB.wdata = ;
			//MEM_WB.regwrite = ;
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction PFXIO (Prefix with Cache Bypass) from format i11.
 * status: finished, tested [Amended]
 */
void ac_behavior ( pfxio ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\t\tpfxio\n");
			break;
		case id_pipe_ID:
			
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			//EX_MEM.wdata = ;
			//EX_MEM.rdest = ;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;

			//control registers
			k.k = ID_EX.imm;

			break;
		case id_pipe_MEM:
			//MEM_WB.rdest = ;
			//MEM_WB.wdata = ;
			//MEM_WB.regwrite = ;
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction RDCTL (Read Control Register) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( rdctl ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\trdctl\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction RESTORE (Restore Caller's Register Window) from format w
 * status: not finished, not tested
 */
void ac_behavior ( restore ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\trestore\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction RET (Equivalent to JMP %i7) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( ret ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tret\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction RLC (Rotate Left Through Carry) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( rlc ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\trlc\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction RRC (Rotate Right Through Carry) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( rrc ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\trrc\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SAVE (Save Caller's Register Window) from format i8v
 * status: not finished, not tested
 */
void ac_behavior ( save ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tsave\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SEXT16 (Sign Extend 16-bit Value) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( sext16 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tsext16\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SEXT8 (Sign Extend 8-bit Value) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( sext8 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tsext8\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SKP0 (Skip if Register Bit Is 0) from format Ri5
 * status: not finished, not tested
 */
void ac_behavior ( skp0 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tskp0\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SKP1 (Skip If Register Bit Is 1) from format Ri5
 * status: not finished, not tested
 */
void ac_behavior ( skp1 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tskp1\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SKPRNZ (Skip If Register Not Equal To 0) from format Rw
 * status: not finished, not tested [Amended]
 */
void ac_behavior ( skprnz ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tskprnz\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:

			//if(ID_EX.data1 != 0){
			//	if( && IF_ID.opcode == 0x13 || IF_ID.opcode == 0x12){//previows from PFX or PFXIO
			//		ID_EX.npc = ID_EX.npc + 0x0000006;
			//}
			//	else ID_EX.npc = ID_EX.npc + 0x0000004;
			//}

			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SKPRZ (Skip If Register Equals 0) from format Rw
 * status: not finished, not tested [Amended]
 */
void ac_behavior ( skprz ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tskprz\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:

			//if(ID_EX.npc == 0){
			//	if(&& k.k != 0){ID_EX.npc = ID_EX.npc + 0x00000006;}
		//		else ID_EX.npc = ID_EX.npc + 0x0000004;
		//	}
//
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SKPS (Skip On Condition Code) from format i4w
 * status: not finished, not tested
 */
void ac_behavior ( skps ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tskps\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ST (Store 32-bit Data To Memory) from format RR
 * status: not finished, not tested
 */
void ac_behavior ( st ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tst\n");
			break;
		case id_pipe_ID:
			//fowarding for register 1
			if ( MEM_WB.rdest == ra ) {
				cout << "\n\tfowarding ra from WB\n";
				ID_EX.data1 = MEM_WB.wdata;
			}
			else if ( EX_MEM.rdest == ra ) {
				cout << "\n\tfowarding ra from MEM\n";
				ID_EX.data1 = EX_MEM.wdata;
			}
			else {
				cout << "\n\tra from RB\n";
				ID_EX.data1 = RB[ra];
			}
			//ID_EX.imm = ;
			//ID_EX.ra = ;
			ID_EX.rb = rb;
			ID_EX.regwrite = 0;
			ID_EX.memread = 0;
			ID_EX.memwrite = 1;
			break;
		case id_pipe_EX:
			if(k.k == 0){
				//EX_MEM.imm = ;
				EX_MEM.wdata = ID_EX.ra;
				EX_MEM.rdest = ID_EX.rb & 0x8000000;
				EX_MEM.regwrite = ID_EX.regwrite;
				EX_MEM.memread = ID_EX.memread;
				EX_MEM.memwrite = ID_EX.memwrite;
			}else{
				//EX_MEM.imm = ;
				EX_MEM.wdata = ID_EX.ra;
				EX_MEM.rdest = ((ID_EX.rb + k.k * 4) & 0x8000000);
				EX_MEM.regwrite = ID_EX.regwrite;
				EX_MEM.memread = ID_EX.memread;
				EX_MEM.memwrite = ID_EX.memwrite;
			}
			break;
		case id_pipe_MEM:
			MEM_WB.wdata = EX_MEM.rdest;			
			break;
		case id_pipe_WB:
			MEM_WB.wdata = MEM_WB.wdata;
			break;
	}

};

/** Instruction ST16D (Store 16-bit Data To Memory (Computed Half-Word Pointer Address)) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( st16d ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tst16d\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ST16S (Store 16-bit Data To Memory (Static Half-Word-Offset Address)) from format Ri1u
 * status: not finished, not tested
 */
void ac_behavior ( st16s ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tst16s\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ST8D (Store 8-bit Data To Memory (Computed Byte-Pointer Address)) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( st8d ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tst8d\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction ST8S (Store 8-bit Data To Memory (Static Byte-Offset Address)) from format Ri2u
 * status: not finished, not tested
 */
void ac_behavior ( st8s ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tst8s\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction STP (Store 32-bit Data To Memory (Pointer Addressing Mode)) from format RPi5
 * status: not finished, not tested
 */
void ac_behavior ( stp ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tstp\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction STS (Store 32-bit Data To Memory (Stack Addressing Mode)) from format Ri8
 * status: not finished, not tested
 */
void ac_behavior ( sts ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tsts\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction STS16S (Store 16-bit Data To Memory (Stack Addressing Mode)) from format i9
 * status: not finished, not tested
 */
void ac_behavior ( sts16s ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tsts16s\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction STS8S (Store 8-bit Data To Memory (Stack Addressing Mode)) from format i10
 * status: not finished, not tested
 */
void ac_behavior ( sts8s ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tsts8s\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SUB (Subtract) from format RR
 * status: finished, not tested [Amended]
 */
void ac_behavior ( sub ) {
	
	int temp, carry;
	
	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tsub\n");
			break;
		case id_pipe_ID:
			ID_EX.npc = IF_ID.npc;

			//fowarding for register 1
			if ( MEM_WB.rdest == ra ) {
				cout << "\n\tfowarding ra from WB\n";
				ID_EX.data1 = MEM_WB.wdata;
			}
			else if ( EX_MEM.rdest == ra ) {
				cout << "\n\tfowarding ra from MEM\n";
				ID_EX.data1 = EX_MEM.wdata;
			}
			else {
				cout << "\n\tra from RB\n";
				ID_EX.data1 = RB[ra];
			}

			//fowarding for register 2
			if ( MEM_WB.rdest == rb ) {
				cout << "\n\tfowarding rb from WB\n";
				ID_EX.data2 = MEM_WB.wdata;
			}
			else if ( EX_MEM.rdest == rb ) {
				cout << "\n\tfowarding rb from MEM\n";
				ID_EX.data2 = EX_MEM.wdata;
			}
			else {
				cout << "\n\trb from RB\n";
				ID_EX.data2 = RB[rb];
			}

			//ID_EX.imm = 0;
			ID_EX.ra = ra;
			ID_EX.rb = rb;
			ID_EX.regwrite = 1;
			ID_EX.memread = 0;
			ID_EX.memwrite = 0;
			
			break;
		case id_pipe_EX:
			//EX_MEM.imm = ;
			EX_MEM.wdata = ID_EX.rb - ID_EX.ra;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = 1;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			
			//control registers
			temp = EX_MEM.wdata & 0x80000000;
			temp = temp >> 31;

			if ( temp == 1 )
				ctl0.n = 1;
			else
				ctl0.n = 0;

			if ( (EX_MEM.wdata & 0x80000000) == 1 )
				ctl0.v = 1;
			else 
				ctl0.v = 0;

			if ( EX_MEM.wdata == 0 ) 
				ctl0.z = 1;
			else 
				ctl0.z = 0;
	
			carry = EX_MEM.wdata >> 31;
						
			if( carry == 1) 
				ctl0.c = 1;
			else 
				ctl0.c = 0;
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SUBI (Subtract Immediate) from format Ri5
 * status: finished, not tested [Amended]
 */
void ac_behavior ( subi ) {

int temp, overflow, carry;
	
	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tsubi\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			if(k.k == 0){//not above PFX or PFXIO
			EX_MEM.imm = ID_EX.imm;
			EX_MEM.wdata = ID_EX.ra - ID_EX.imm;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			}
			else{//above PFX or PFXIO
			EX_MEM.imm = ID_EX.imm;
			EX_MEM.wdata = ID_EX.ra - ( ID_EX.imm | (k.k<<5) );
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;
			}
			
			//control register
			temp = EX_MEM.wdata & 0x80000000;
			temp = temp >> 31;

			if(temp == 1) ctl0.n = 1;
			else ctl0.n = 0;
		
			overflow = EX_MEM.wdata & 0x8000000;			

			if(overflow == 1) ctl0.v = 1;
			else ctl0.v = 0;
			
			if(EX_MEM.wdata != 0) ctl0.z = 1;
			else ctl0.z = 0;
			
			carry = EX_MEM.wdata >> 31;
						
			if( carry == 1) 
				ctl0.c = 1;
			else 
				ctl0.c = 0;
		break;

		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SWAP (Swap Register Half-Words) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( swap ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tswap\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction TRAP (Unconditional Trap) from format i6v
 * status: not finished, not tested [Amended]
 */
void ac_behavior ( trap ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\ttrap\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/* The value in IStatus copied into the Status */
			ctl1.savedstatus = ctl0.dc;
			ctl1.savedstatus = ctl0.ic;
			ctl1.savedstatus = ctl0.ie;
			ctl1.savedstatus = ctl0.ipri;
			ctl1.savedstatus = ctl0.cwp;
			ctl1.savedstatus = ctl0.n;
			ctl1.savedstatus = ctl0.v;
			ctl1.savedstatus = ctl0.z;
			ctl1.savedstatus = ctl0.c;

			ctl0.ie = 0;
			ctl0.cwp = ctl0.cwp - 1;
			ctl0.ipri = ID_EX.imm;
			RB[15] = ((ac_pc + 2) >> 1);
			//ac_pc = ((ID_EX.imm * 4)) << 1) & 0x80000000;


			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction TRET (Trap Return) from format Rw
 * status: finished, not tested [Amended]
 */
void ac_behavior ( tret ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\ttret\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:

			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/

			/* PC recebe o deslocamento a esquerda de 1bit em RA */
			ac_pc = ID_EX.data1 << 1;
	
			/* The value in Status copied into the IStatus */
			ctl0.dc = ctl1.savedstatus;
			ctl0.ic = ctl1.savedstatus;
			ctl0.ie = ctl1.savedstatus;
			ctl0.ipri = ctl1.savedstatus;
			ctl0.cwp = ctl1.savedstatus;
			ctl0.n = ctl1.savedstatus;
			ctl0.v = ctl1.savedstatus;
			ctl0.z = ctl1.savedstatus;
			ctl0.c = ctl1.savedstatus;
			break;
		case id_pipe_MEM:
			cout << "\t\n tret -> PC" << ac_pc;
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction USR0 (User-defined Instruction) from format RR
 * status: not finished, not tested
 */
void ac_behavior ( usr0 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tusr0\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction USR1 (User-defined Instruction) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( usr1 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tusr1\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}
};

/** Instruction USR2 (User-defined Instruction) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( usr2 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tusr2\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}
};

/** Instruction USR3 (User-defined Instruction) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( usr3 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tusr3\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction USR4 (User-defined Instruction) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( usr4 ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\tusr4\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction WRCTL (Write Control Register) from format Rw
 * status: not finished, not tested
 */
void ac_behavior ( wrctl ) {

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\twrctl\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			/*EX_MEM.imm = ;
			EX_MEM.wdata = ;
			EX_MEM.rdest = ;
			EX_MEM.regwrite = ;
			EX_MEM.memread = ;
			EX_MEM.memwrite = ;*/
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction XOR (Bitwise Logical Exclusive OR) not preceded by PFX from format RR
 * status: finished, not tested [Amended]
 */
void ac_behavior ( xor1 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\txor not preceded by pfx\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:

			//EX_MEM.imm = ;
			EX_MEM.wdata = ID_EX.ra ^ ID_EX.rb;
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;

			temp = EX_MEM.wdata & 0x8000000;
			temp = temp >> 31;

			if( temp == 1 )
				ctl0.n = 1;
			else 
				ctl0.n = 0;

			if( EX_MEM.wdata == 0 )
				ctl0.z = 1;
			else
				ctl0.z = 0;
 
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction XOR (Bitwise Logical Exclusive OR) preceded by PFX from format Ri5
 * status: finished, not tested [Amended]
 */
void ac_behavior ( xor2 ) {

	int temp;

	switch ( stage ) {
		case id_pipe_IF:
			printf ("\n\t\txor preceded by pfx\n");
			break;
		case id_pipe_ID:
			break;
		case id_pipe_EX:
			EX_MEM.imm = ID_EX.imm;
			EX_MEM.wdata = ID_EX.ra ^ ( ID_EX.imm | (k.k << 5) );
			EX_MEM.rdest = ID_EX.ra;
			EX_MEM.regwrite = ID_EX.regwrite;
			EX_MEM.memread = 0;
			EX_MEM.memwrite = 0;

			temp = EX_MEM.wdata & 0x8000000;
			temp = temp >> 31;

			if( temp == 1 )
				ctl0.n = 1;
			else	ctl0.n = 0;

			if( EX_MEM.wdata == 0 )
				ctl0.z = 1;
			else	ctl0.z = 0;		
	
			break;
		case id_pipe_MEM:
			break;
		case id_pipe_WB:
			break;
	}

};

/** Instruction SYSCALL (not in NIOS)
 * not working
 */
void ac_behavior ( syscall ) {
	return;
}
